import { ReactNode } from "react";
import { Switch, Route } from "wouter";
import { Toaster } from "@/components/ui/toaster";
import NotFound from "@/pages/not-found";
import HomePage from "@/pages/home-page";
import AuthPage from "@/pages/auth-page";
import MovieDetails from "@/pages/movie-details";
import TranslatedMovies from "@/pages/translated-movies";
import NonTranslatedMovies from "@/pages/non-translated-movies";
import AdminDashboard from "@/pages/admin/dashboard";
import AdminMovieManagement from "@/pages/admin/movie-management";
import AdminUserManagement from "@/pages/admin/user-management";
import AdminUploadMovie from "@/pages/admin/upload-movie";
import { ProtectedRoute } from "@/lib/protected-route";
import { AuthProvider } from "@/hooks/use-auth";

// Layout wrapper for routes that need auth context
const AuthLayout = ({ children }: { children: ReactNode }) => {
  return (
    <AuthProvider>
      {children}
    </AuthProvider>
  );
};

function App() {
  return (
    <div className="min-h-screen bg-dark text-text-light font-sans">
      <Switch>
        {/* Public Routes with Auth */}
        <Route path="/">
          <AuthLayout>
            <HomePage />
          </AuthLayout>
        </Route>
        <Route path="/auth">
          <AuthLayout>
            <AuthPage />
          </AuthLayout>
        </Route>
        <Route path="/movies/:id">
          <AuthLayout>
            <MovieDetails />
          </AuthLayout>
        </Route>
        <Route path="/translated-movies">
          <AuthLayout>
            <TranslatedMovies />
          </AuthLayout>
        </Route>
        <Route path="/non-translated-movies">
          <AuthLayout>
            <NonTranslatedMovies />
          </AuthLayout>
        </Route>
        
        {/* Admin Routes */}
        <Route path="/admin">
          <AuthLayout>
            <ProtectedRoute path="/admin" component={AdminDashboard} adminOnly={true} />
          </AuthLayout>
        </Route>
        <Route path="/admin/movies">
          <AuthLayout>
            <ProtectedRoute path="/admin/movies" component={AdminMovieManagement} adminOnly={true} />
          </AuthLayout>
        </Route>
        <Route path="/admin/users">
          <AuthLayout>
            <ProtectedRoute path="/admin/users" component={AdminUserManagement} adminOnly={true} />
          </AuthLayout>
        </Route>
        <Route path="/admin/upload">
          <AuthLayout>
            <ProtectedRoute path="/admin/upload" component={AdminUploadMovie} adminOnly={true} />
          </AuthLayout>
        </Route>
        
        {/* Fallback to 404 */}
        <Route>
          <NotFound />
        </Route>
      </Switch>
      <Toaster />
    </div>
  );
}

export default App;
