import { useState } from "react";
import { Link } from "wouter";
import { Movie } from "@shared/schema";
import { Button } from "@/components/ui/button";
import { PlayCircle, Download } from "lucide-react";

interface MovieCardProps {
  movie: Movie;
  compact?: boolean;
}

export default function MovieCard({ movie, compact = false }: MovieCardProps) {
  const [isHovered, setIsHovered] = useState(false);

  const handleDownload = (e: React.MouseEvent) => {
    e.preventDefault();
    // In a real app, we'd make a request to increment the download count
    // and then initiate the download
    window.open(movie.videoUrl, "_blank");
  };

  return (
    <div 
      className={`movie-card relative overflow-hidden rounded-lg shadow-lg transition-transform duration-300 ${compact ? "" : "w-64"}`}
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
    >
      <Link href={`/movies/${movie.id}`}>
        <img 
          src={movie.posterUrl} 
          className={`w-full ${compact ? "h-64" : "h-96"} object-cover transition-transform duration-300 ${isHovered ? "scale-105" : ""}`} 
          alt={movie.title} 
        />
        <div className={`absolute inset-0 bg-gradient-to-t from-black via-transparent to-transparent ${isHovered ? "opacity-100" : "opacity-0"} transition-opacity duration-300 flex flex-col justify-end p-4`}>
          <h3 className={`${compact ? "text-lg" : "text-xl"} font-bold text-white`}>{movie.title}</h3>
          
          {movie.isTranslated && movie.vjName && (
            <p className="text-sm text-gray-300">VJ: {movie.vjName}</p>
          )}
          
          <div className="flex items-center mt-2">
            <span className="bg-primary text-xs text-white px-2 py-1 rounded mr-2">{movie.genre}</span>
            <span className="bg-gray-700 text-xs text-white px-2 py-1 rounded">{movie.language}</span>
          </div>
          
          {!compact && (
            <div className="mt-4 flex space-x-2">
              <Button 
                size="sm" 
                className="bg-primary hover:bg-secondary text-white text-sm px-3 py-1 rounded flex items-center"
              >
                <PlayCircle className="h-4 w-4 mr-1" /> Play
              </Button>
              
              <Button 
                size="sm" 
                className="bg-gray-700 hover:bg-gray-600 text-white text-sm px-3 py-1 rounded flex items-center"
                onClick={handleDownload}
              >
                <Download className="h-4 w-4 mr-1" /> Download
              </Button>
            </div>
          )}
        </div>
      </Link>
    </div>
  );
}
