import { useState } from "react";
import { Link, useLocation } from "wouter";
import { 
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { Search, Bell, Menu, User } from "lucide-react";
import { User as UserType } from "@shared/schema";

interface NavbarProps {
  onSearch?: (query: string) => void;
  user?: UserType | null;
  onLogout?: () => void;
}

export default function Navbar({ onSearch, user, onLogout }: NavbarProps) {
  const [location] = useLocation();
  const [searchQuery, setSearchQuery] = useState("");
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    if (onSearch) {
      onSearch(searchQuery);
    }
  };

  const handleLogout = () => {
    if (onLogout) {
      onLogout();
    }
  };

  const getInitials = (name: string) => {
    return name.charAt(0).toUpperCase();
  };

  return (
    <nav className="bg-dark-card shadow-lg">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between h-16">
          <div className="flex items-center">
            <Link href="/" className="flex-shrink-0 flex items-center">
              <span className="text-primary font-bold text-xl">MovieStream</span>
            </Link>
            <div className="hidden md:ml-6 md:flex md:space-x-8">
              <Link 
                href="/" 
                className={`${location === "/" ? "border-b-2 border-primary text-text-light" : "border-transparent text-gray-300 hover:border-accent hover:text-text-light border-b-2"} px-1 pt-1 text-sm font-medium`}
              >
                Home
              </Link>
              <Link 
                href="/translated-movies" 
                className={`${location === "/translated-movies" ? "border-b-2 border-primary text-text-light" : "border-transparent text-gray-300 hover:border-accent hover:text-text-light border-b-2"} px-1 pt-1 text-sm font-medium`}
              >
                Translated Movies
              </Link>
              <Link 
                href="/non-translated-movies" 
                className={`${location === "/non-translated-movies" ? "border-b-2 border-primary text-text-light" : "border-transparent text-gray-300 hover:border-accent hover:text-text-light border-b-2"} px-1 pt-1 text-sm font-medium`}
              >
                Non-Translated
              </Link>
            </div>
          </div>

          <div className="flex items-center">
            <div className="hidden md:block">
              <form onSubmit={handleSearch} className="relative">
                <Input
                  type="text"
                  placeholder="Search movies, VJs, genres..."
                  className="bg-gray-800 rounded-full px-4 py-2 text-sm text-white focus:outline-none focus:ring-2 focus:ring-primary w-64"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                />
                <Button 
                  type="submit"
                  variant="ghost" 
                  size="icon" 
                  className="absolute right-0 top-0 mt-2 mr-3 text-gray-400 hover:text-gray-300"
                >
                  <Search className="h-4 w-4" />
                </Button>
              </form>
            </div>

            <div className="ml-4 flex items-center md:ml-6">
              <Button variant="ghost" size="icon" className="bg-gray-800 p-1 rounded-full text-gray-400 hover:text-white">
                <Bell className="h-5 w-5" />
              </Button>

              {user ? (
                <div className="ml-3 relative">
                  <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                      <Button variant="ghost" className="flex items-center">
                        <Avatar className="h-8 w-8 bg-primary">
                          <AvatarFallback>{getInitials(user.username)}</AvatarFallback>
                        </Avatar>
                      </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end" className="w-56">
                      <DropdownMenuLabel>My Account</DropdownMenuLabel>
                      <DropdownMenuSeparator />
                      <DropdownMenuItem>
                        <User className="mr-2 h-4 w-4" />
                        <span>Profile</span>
                      </DropdownMenuItem>
                      {user.isAdmin && (
                        <DropdownMenuItem>
                          <Link href="/admin" className="w-full">Admin Dashboard</Link>
                        </DropdownMenuItem>
                      )}
                      <DropdownMenuSeparator />
                      <DropdownMenuItem onClick={handleLogout}>
                        Logout
                      </DropdownMenuItem>
                    </DropdownMenuContent>
                  </DropdownMenu>
                </div>
              ) : (
                <Link href="/auth">
                  <Button className="ml-4 bg-primary hover:bg-secondary text-white px-4 py-2 rounded-md text-sm font-medium">
                    Sign In
                  </Button>
                </Link>
              )}
            </div>

            <div className="-mr-2 flex md:hidden">
              <Button
                variant="ghost"
                onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
                aria-controls="mobile-menu"
                aria-expanded={isMobileMenuOpen}
                className="bg-gray-800 inline-flex items-center justify-center p-2 rounded-md text-gray-400 hover:text-white hover:bg-gray-700"
              >
                <span className="sr-only">Open main menu</span>
                <Menu className="h-6 w-6" />
              </Button>
            </div>
          </div>
        </div>
      </div>

      {isMobileMenuOpen && (
        <div className="md:hidden" id="mobile-menu">
          <div className="px-2 pt-2 pb-3 space-y-1 sm:px-3">
            <Link 
              href="/" 
              className={`${location === "/" ? "bg-gray-900 text-white" : "text-gray-300 hover:bg-gray-700 hover:text-white"} block px-3 py-2 rounded-md text-base font-medium`}
              onClick={() => setIsMobileMenuOpen(false)}
            >
              Home
            </Link>
            <Link 
              href="/translated-movies" 
              className={`${location === "/translated-movies" ? "bg-gray-900 text-white" : "text-gray-300 hover:bg-gray-700 hover:text-white"} block px-3 py-2 rounded-md text-base font-medium`}
              onClick={() => setIsMobileMenuOpen(false)}
            >
              Translated Movies
            </Link>
            <Link 
              href="/non-translated-movies" 
              className={`${location === "/non-translated-movies" ? "bg-gray-900 text-white" : "text-gray-300 hover:bg-gray-700 hover:text-white"} block px-3 py-2 rounded-md text-base font-medium`}
              onClick={() => setIsMobileMenuOpen(false)}
            >
              Non-Translated
            </Link>

            <form onSubmit={handleSearch} className="mt-2">
              <Input
                type="text"
                placeholder="Search movies, VJs, genres..."
                className="bg-gray-800 rounded-full px-4 py-2 text-sm text-white focus:outline-none focus:ring-2 focus:ring-primary w-full"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
              />
              <Button 
                type="submit"
                className="mt-2 w-full bg-primary hover:bg-secondary text-white"
              >
                Search
              </Button>
            </form>
          </div>

          {user && (
            <div className="pt-4 pb-3 border-t border-gray-700">
              <div className="flex items-center px-5">
                <div className="flex-shrink-0">
                  <Avatar className="h-10 w-10 bg-primary">
                    <AvatarFallback>{getInitials(user.username)}</AvatarFallback>
                  </Avatar>
                </div>
                <div className="ml-3">
                  <div className="text-base font-medium leading-none text-white">{user.username}</div>
                  <div className="text-sm font-medium leading-none text-gray-400">{user.email}</div>
                </div>
              </div>
              <div className="mt-3 px-2 space-y-1">
                <Link 
                  href="/profile" 
                  className="block px-3 py-2 rounded-md text-base font-medium text-gray-400 hover:text-white hover:bg-gray-700"
                  onClick={() => setIsMobileMenuOpen(false)}
                >
                  Your Profile
                </Link>
                {user.isAdmin && (
                  <Link 
                    href="/admin" 
                    className="block px-3 py-2 rounded-md text-base font-medium text-gray-400 hover:text-white hover:bg-gray-700"
                    onClick={() => setIsMobileMenuOpen(false)}
                  >
                    Admin Dashboard
                  </Link>
                )}
                <button
                  onClick={handleLogout}
                  className="block w-full text-left px-3 py-2 rounded-md text-base font-medium text-gray-400 hover:text-white hover:bg-gray-700"
                >
                  Sign out
                </button>
              </div>
            </div>
          )}
        </div>
      )}
    </nav>
  );
}
